﻿; https://www.autohotkey.com/board/topic/77303-acc-library-ahk-l-updated-09272012/
; https://www.autohotkey.com/boards/viewtopic.php?f=7&t=40590
;------------------------------------------------------------------------------
; Acc.ahk Standard Library
; by Sean
; Updated by jethrow:
;	Modified ComObjEnwrap params from (9,pacc) --> (9,pacc,1)
;	Changed ComObjUnwrap to ComObjValue in order to avoid AddRef (thanks fincs)
;	Added Acc_GetRoleText & Acc_GetStateText
;	Added additional functions - commented below
;	Removed original Acc_Children function
;	Added Acc_Error, Acc_ChildrenByRole, & Acc_Get functions
; Updated by jeeswg (for AHK v2.0+)
; last updated 2023-04-19
;------------------------------------------------------------------------------

#Requires AutoHotkey v2+
;for AutoHotkey v1.1.35+:
;replace '&' with 'ByRef ' (in Acc_ObjectFromEvent/Acc_ObjectFromPoint/Acc_Location)
;use backport libraries (e.g. JEEAhk1FC.ahk/JEEAhk1FCGui.ahk)

Acc_Init()
{
	static h := 0
	if !h
		h := DllCall("kernel32\LoadLibrary", "Str","oleacc", "Ptr")
}

Acc_ObjectFromEvent(&_idChild_, hWnd, idObject, idChild)
{
	local pacc, varChild
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Acc_Init()
	pacc := 0
	varChild := Buffer(8+2*A_PtrSize, 0)
	if !DllCall("oleacc\AccessibleObjectFromEvent", "Ptr",hWnd, "UInt",idObject, "UInt",idChild, "Ptr*",IsV1?pacc:&pacc, "Ptr",varChild.Ptr)
	{
		_idChild_ := NumGet(varChild.Ptr, 8, "UInt")
		return ComObjFromPtr(pacc)
	}
}

Acc_ObjectFromPoint(&_idChild_:="", x:="", y:="")
{
	local pacc, pt, varChild
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Acc_Init()
	pt := pacc := 0
	varChild := Buffer(8+2*A_PtrSize, 0)
	if !DllCall("oleacc\AccessibleObjectFromPoint", "UInt64",x == "" || y == "" ? 0*DllCall("user32\GetCursorPos", "Int64*",IsV1?pt:&pt)+pt : x & 0xFFFFFFFF | y << 32, "Ptr*",IsV1?pacc:&pacc, "Ptr",varChild.Ptr)
	{
		_idChild_ := NumGet(varChild.Ptr, 8, "UInt")
		return ComObjFromPtr(pacc)
	}
}

Acc_ObjectFromWindow(hWnd, idObject:=-4)
{
	local IID, pacc
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	Acc_Init()

	static IID_IDispatch := "{00020400-0000-0000-C000-000000000046}"
	static IID_IAccessible := "{618736E0-3C3D-11CF-810C-00AA00389B71}"
	IID := Buffer(16)
	if (idObject & 0xFFFFFFFF = 0xFFFFFFF0) ;OBJID_NATIVEOM := -16 ;(LONG)0xFFFFFFF0
		DllCall("ole32\CLSIDFromString", "WStr",IID_IDispatch, "Ptr",IID.Ptr)
	else
		DllCall("ole32\CLSIDFromString", "WStr",IID_IAccessible, "Ptr",IID.Ptr)

	pacc := 0
	if !DllCall("oleacc\AccessibleObjectFromWindow", "Ptr",hWnd, "UInt",idObject, "Ptr",IID.Ptr, "Ptr*",IsV1?pacc:&pacc)
		return ComObjFromPtr(pacc)
}

Acc_WindowFromObject(pacc)
{
	local hWnd
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	static IID_IAccessible := "{618736E0-3C3D-11CF-810C-00AA00389B71}"
	hWnd := 0
	try
	{
		if !DllCall("oleacc\WindowFromAccessibleObject", "Ptr",IsObject(pacc) ? ComObjValue(pacc) : pacc, "Ptr*",IsV1?hWnd:&hWnd)
			return hWnd
	}
	try
	{
		;appears to be sometimes necessary in AHK v2, but not AHK v1:
		pacc := ComObjQuery(pacc, IID_IAccessible)
		if !DllCall("oleacc\WindowFromAccessibleObject", "Ptr",IsObject(pacc) ? ComObjValue(pacc) : pacc, "Ptr*",IsV1?hWnd:&hWnd)
			return hWnd
	}
	return 0
}

Acc_GetRoleText(nRole)
{
	local nSize, sRole
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	nSize := DllCall("oleacc\GetRoleText", "UInt",nRole, "Ptr",0, "UInt",0, "UInt")
	VarSetStrCapacity(IsV1?sRole:&sRole, nSize)
	DllCall("oleacc\GetRoleText", "UInt",nRole, "Str",sRole, "UInt",nSize+1, "UInt")
	return sRole
}

Acc_GetStateText(nState)
{
	local nSize, sState
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	nSize := DllCall("oleacc\GetStateText", "UInt",nState, "Ptr",0, "UInt",0, "UInt")
	VarSetStrCapacity(IsV1?sState:&sState, nSize)
	DllCall("oleacc\GetStateText", "UInt",nState, "Str",sState, "UInt",nSize+1, "UInt")
	return sState
}

Acc_SetWinEventHook(eventMin, eventMax, pCallback)
{
	return DllCall("user32\SetWinEventHook", "UInt",eventMin, "UInt",eventMax, "Ptr",0, "Ptr",pCallback, "UInt",0, "UInt",0, "UInt",0, "Ptr")
}

Acc_UnhookWinEvent(hHook)
{
	return DllCall("user32\UnhookWinEvent", "Ptr",hHook)
}

;/* Win Events :
;	pCallback := CallbackCreate(WinEventProc)
;	WinEventProc(hHook, event, hWnd, idObject, idChild, eventThread, eventTime)
;	{
;		Critical()
;		;AHK v1: Acc := Acc_ObjectFromEvent(_idChild_, hWnd, idObject, idChild)
;		Acc := Acc_ObjectFromEvent(&_idChild_, hWnd, idObject, idChild)
;		; Code Here:
;
;	}
;*/

; Written by jethrow
Acc_Role(Acc, ChildID:=0)
{
	try return ComObjType(Acc, "Name") = "IAccessible" ? Acc_GetRoleText(Acc.accRole[ChildID]) : "invalid object"
}
Acc_State(Acc, ChildID:=0)
{
	try return ComObjType(Acc, "Name") = "IAccessible" ? Acc_GetStateText(Acc.accState[ChildID]) : "invalid object"
}
Acc_Location(Acc, ChildID:=0, &Position:="") ; adapted from Sean's code
{
	local pos, RECT
	RECT := Buffer(16, 0)
	try Acc.accLocation(ComValue(0x4003, RECT.Ptr), ComValue(0x4003, RECT.Ptr+4), ComValue(0x4003, RECT.Ptr+8), ComValue(0x4003, RECT.Ptr+12), ChildID)
	catch
	{
		Position := "x  y  w  h "
		return {x:"", y:"", w:"", h:"", pos:Position}
	}
	Position := "x" NumGet(RECT.Ptr, 0, "Int") " y" NumGet(RECT.Ptr, 4, "Int") " w" NumGet(RECT.Ptr, 8, "Int") " h" NumGet(RECT.Ptr, 12, "Int")
	return {x:NumGet(RECT.Ptr, 0, "Int"), y:NumGet(RECT.Ptr, 4, "Int"), w:NumGet(RECT.Ptr, 8, "Int"), h:NumGet(RECT.Ptr, 12, "Int"), pos:Position}
}
Acc_Parent(Acc)
{
	local parent
	try parent := Acc.accParent
	catch
		parent := ""
	return parent ? Acc_Query(parent) : ""
}
Acc_Child(Acc, ChildID:=0)
{
	local child
	child := 0
	try child := Acc.accChild[ChildID]
	return child ? Acc_Query(child) : ""
}
Acc_Query(Acc) ; thanks Lexikos - https://www.autohotkey.com/board/topic/76309-closed-help-with-acc-viewer-project/?p=486291
{
	;AHK v1: try return ComObject(9, ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}"), 1) ;IID_IAccessible
	;AHK v1: try return ComValue(9, ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}"), 1) ;IID_IAccessible

	local pAcc
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	try
	{
		pAcc := ComObjQuery(Acc, "{618736e0-3c3d-11cf-810c-00aa00389b71}") ;IID_IAccessible
		ObjAddRef(IsV1?pAcc:pAcc.Ptr)
		return ComValue(9, IsV1?pAcc:pAcc.Ptr) ;VT_DISPATCH := 9
	}
}
Acc_Error(p:="")
{
	static setting := 0
	return p = "" ? setting : setting := p
}
Acc_Children(Acc)
{
	local cChildren, Child, Children, ErrMsg, i, varChildren
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if ComObjType(Acc, "Name") != "IAccessible"
		ErrMsg := "Invalid IAccessible Object"
	else
	{
		Acc_Init()
		cChildren := Acc.accChildCount
		Children := []
		if !cChildren
			return Children ;update: the function now returns an empty array if there are no children (it used to return a blank string)
		varChildren := Buffer(cChildren*(8+2*A_PtrSize), 0)
		if !DllCall("oleacc\AccessibleChildren", "Ptr",ComObjValue(Acc), "Int",0, "Int",cChildren, "Ptr",varChildren.Ptr, "Int*",IsV1?cChildren:&cChildren)
		{
			;AHK v1: Loop % cChildren
			while (A_Index <= cChildren)
			{
				i := (A_Index-1)*(A_PtrSize*2+8) + 8
				if (NumGet(varChildren.Ptr, i-8, "UShort") == 9) ;child object ;VT_DISPATCH := 9
				{
					Child := NumGet(varChildren.Ptr, i, "UPtr")
					Children.Push(Acc_Query(Child))
					ObjRelease(Child)
				}
				else ;child ID ;VT_I4 := 3
					Children.Push(NumGet(varChildren.Ptr, i, "Int"))
			}
			return Children
		}
		ErrMsg := "AccessibleChildren DllCall Failed"
	}
	if Acc_Error()
		throw Error(ErrMsg, -1)
}
Acc_ChildrenByRole(Acc, Role)
{
	local AccChild, cChildren, Child, Children, ErrMsg, i, varChildren
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	if ComObjType(Acc, "Name") != "IAccessible"
		ErrMsg := "Invalid IAccessible Object"
	else
	{
		Acc_Init()
		cChildren := Acc.accChildCount
		Children := []
		if !cChildren
			return Children ;update: the function now returns an empty array if there are no children (it used to return a blank string)
		varChildren := Buffer(cChildren*(8+2*A_PtrSize), 0)
		if !DllCall("oleacc\AccessibleChildren", "Ptr",ComObjValue(Acc), "Int",0, "Int",cChildren, "Ptr",varChildren.Ptr, "Int*",IsV1?cChildren:&cChildren)
		{
			;AHK v1: Loop % cChildren
			while (A_Index <= cChildren)
			{
				i := (A_Index-1)*(A_PtrSize*2+8) + 8
				if (NumGet(varChildren.Ptr, i-8, "UShort") == 9) ;child object ;VT_DISPATCH := 9
				{
					Child := NumGet(varChildren.Ptr, i, "UPtr")
					AccChild := Acc_Query(Child)
					ObjRelease(Child)
					(Acc_Role(AccChild) = Role) && Children.Push(AccChild)
				}
				else ;child ID ;VT_I4 := 3
				{
					Child := NumGet(varChildren.Ptr, i, "Int")
					(Acc_Role(Acc, Child) = Role) && Children.Push(Child)
				}
			}
			return Children
		}
		ErrMsg := "AccessibleChildren DllCall Failed"
	}
	if Acc_Error()
		throw Error(ErrMsg, -1)
}
Acc_Get(Cmd, ChildPath:="", ChildID:=0, WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local AccError, AccObj, Children, ErrMsg, m, m2, RECT, ret_val
	local Loop1, Loop2, Loop1X, Loop2X
	static properties := Map("Action","DefaultAction", "DoAction","DoDefaultAction", "Keyboard","KeyboardShortcut")
	static IsV1 := (InStr(A_AhkVersion, "1.") == 1)
	AccObj := IsObject(WinTitle) ? WinTitle
	: Acc_ObjectFromWindow(WinExist(WinTitle, WinText, ExcludeTitle, ExcludeText), 0)
	if ComObjType(AccObj, "Name") != "IAccessible"
		ErrMsg := "Could not access an IAccessible Object"
	else
	{
		ChildPath := StrReplace(ChildPath, "_", " ")
		AccError := Acc_Error()
		Acc_Error(true)
		;AHK v1: Loop Parse, ChildPath, % ".", % " "
		Loop1 := "."
		Loop2 := " "
		Loop1X := IsV1 ? Loop1 : "Loop1"
		Loop2X := IsV1 ? Loop2 : "Loop2"
		Loop Parse, ChildPath, %Loop1X%, %Loop2X%
		{
			try
			{
				if IsDigit(A_LoopField)
				{
					Children := Acc_Children(AccObj)
					m2 := A_LoopField ; mimic "m2" output in else-statement
				}
				else
				{
					if IsV1
						RegExMatch(A_LoopField, "O)(\D*)(\d*)", m)
					else
						RegExMatch(A_LoopField, "(\D*)(\d*)", &m)
					Children := Acc_ChildrenByRole(AccObj, m[1])
					m2 := m[2] ? m[2] : 1
				}
				if IsV1 && !Children.HasKey(m2)
					throw
				if !IsV1 && !Children.Has(m2)
					throw
				AccObj := Children[m2]
			}
			catch
			{
				ErrMsg := "Cannot access ChildPath Item #" A_Index " -> " A_LoopField
				Acc_Error(AccError)
				if Acc_Error()
					throw Error("Cannot access ChildPath Item", -1, "Item #" A_Index " -> " A_LoopField)
				return
			}
		}
		Acc_Error(AccError)
		Cmd := StrReplace(Cmd, " ")
		if IsV1
			properties.HasKey(Cmd) && (Cmd := properties[Cmd])
		else
			properties.Has(Cmd) && (Cmd := properties[Cmd])
		try
		{
			if (Cmd = "Location")
			{
				RECT := Buffer(16, 0)
				AccObj.accLocation(ComValue(0x4003, RECT.Ptr), ComValue(0x4003, RECT.Ptr+4), ComValue(0x4003, RECT.Ptr+8), ComValue(0x4003, RECT.Ptr+12), ChildID)
				ret_val := "x" NumGet(RECT.Ptr, 0, "Int") " y" NumGet(RECT.Ptr, 4, "Int") " w" NumGet(RECT.Ptr, 8, "Int") " h" NumGet(RECT.Ptr, 12, "Int")
			}
			else if (Cmd = "Object")
				ret_val := AccObj
			else if (Cmd = "Role") || (Cmd = "State")
				ret_val := Acc_%Cmd%(AccObj, ChildID+0)
			else if (Cmd = "ChildCount")
				ret_val := AccObj.accChildCount
			else if (Cmd = "Selection")
				ret_val := AccObj.accSelection
			else if (Cmd = "Focus")
				ret_val := AccObj.accFocus
			else if (Cmd = "Parent")
				ret_val := AccObj.accParent
			else if (Cmd = "DoDefaultAction")
				ret_val := AccObj.accDoDefaultAction(ChildID+0) ;note: '()' not '[]' (method, not property)
			else if (Cmd = "Child")
				ret_val := AccObj.accChild[ChildID+0]
			else if (Cmd = "DefaultAction")
				ret_val := AccObj.accDefaultAction[ChildID+0]
			else if (Cmd = "Description")
				ret_val := AccObj.accDescription[ChildID+0]
			else if (Cmd = "Help")
				ret_val := AccObj.accHelp[ChildID+0]
			else if (Cmd = "KeyboardShortcut")
				ret_val := AccObj.accKeyboardShortcut[ChildID+0]
			else if (Cmd = "Name")
				ret_val := AccObj.accName[ChildID+0]
			else if (Cmd = "Value")
				ret_val := AccObj.accValue[ChildID+0]
			;not handled: accHitTest()/accNavigate()/accSelect()/accHelpTopic[]
		}
		catch
		{
			ErrMsg := Chr(34) Cmd Chr(34) " Cmd Not Implemented"
			if Acc_Error()
				throw Error("Cmd Not Implemented", -1, Cmd)
			return
		}
		return ret_val
	}
	if Acc_Error()
		throw Error(ErrMsg, -1)
}
